<?php

// Require libraries needed for gateway module functions.
require_once __DIR__ . '/../../../init.php';
require_once __DIR__ . '/../../../includes/gatewayfunctions.php';
require_once __DIR__ . '/../../../includes/invoicefunctions.php';

function errorResponse($message)
{
    global $gateway_params;

    logTransaction($gateway_params['name'], $_REQUEST, $message);
    header("HTTP/1.1 400 Bad Request");
    echo json_encode(["message" => $message]);
    exit;
}

function getCurrentPayments($transactions)
{
    global $gateway_params;

    $payments = [];
    $refunds = [];
    $refund_prefix = "refund-";

    foreach ($transactions as $transaction) {
        if ($transaction["gateway"] !== $gateway_params['name']) {
            continue;
        }

        if (strpos($transaction["transid"], $refund_prefix) === 0) {
            $refunds[] = substr($transaction["transid"], strlen($refund_prefix));
        } else {
            $payments[] = $transaction["transid"];
        }
    }

    foreach ($refunds as $refund) {
        if (($key = array_search($refund, $payments)) !== false) {
            unset($payments[$key]);
        }
    }

    return $payments;
}

// Detect module name from filename.
$gateway_module_name = basename(__FILE__, '.php');

// Fetch gateway configuration parameters.
$gateway_params = getGatewayVariables($gateway_module_name);

// Die if module is not active.
if (!$gateway_params['type']) {
    errorResponse("Módulo desactivado");
}

/**
 * Validate callback authenticity.
 */
switch ($_SERVER["REQUEST_METHOD"]) {
    case "GET":
        $content = html_entity_decode($_GET["encoded_data"]);
        break;
    case "POST":
        $content = file_get_contents('php://input');
        break;
}

$local_verification_hash = hash_hmac("sha256", $content, $gateway_params['secret_key']);
if ($_SERVER["HTTP_X_LINKIFY_CONFIRMATION"] !== $local_verification_hash) {
    errorResponse("Ha fallado la verificación de la firma de la solicitud");
}

// Get request data
$request_data = json_decode($content, true);
if (!$request_data) {
    errorResponse("El formato del contenido no es un JSON válido");
}

$invoice_id = $request_data['id'];

$invoice_data = localAPI('GetInvoice', ['invoiceid' => $request_data['id']]);
if ($invoice_data["result"] !== "success") {
    errorResponse("Ha ocurrido un error en la llamada de la API (GetInvoice)");
}

$balance = round(doubleval($invoice_data["balance"]));

$client_data = localAPI('GetClientsDetails', ['clientid' => $invoice_data['userid']]);
if ($client_data["result"] !== "success") {
    errorResponse("Ha ocurrido un error en la llamada de la API (GetClientsDetails)");
}

if ($_SERVER["REQUEST_METHOD"] == "GET") {
    $description = implode('<br>', array_column($invoice_data["items"]["item"], 'description'));

    if ($balance <= 0) {
        errorResponse("Esta nota de venta no presenta deuda");
    }

    $response = [
        "description" => $description,
        "amount" => $balance,
        "currency" => $client_data["client"]["currency_code"]
    ];
} elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
    $action = $request_data['action'];
    if ($action === "notification") {
        $rejected = false;
        if ($client_data["client"]["currency_code"] !== $request_data["currency"]) {
            $rejected = "Currency code mismatch";
        }

        // This must be done for idempotency
        $current_payments = getCurrentPayments($invoice_data["transactions"]["transaction"]);
        $existing_transfers = array_filter($request_data["transfers"], function ($transfer) use ($current_payments) {
            return in_array($transfer["hashid"], $current_payments);
        });
        $total_paid_by_existing_transfers = array_reduce($existing_transfers, function ($total_paid, $transfer) {
            return $total_paid + $transfer["amount"];
        }, 0);
        $real_balance = $balance + $total_paid_by_existing_transfers;

        if ($real_balance != $request_data["original_amount"]) {
            $rejected = "Se ha modificado la nota de venta. Por favor reinicie el proceso para actualizarla.";
        }

        if ($rejected) {
            $response = [
                "status" => "rejected",
                "message" => $rejected
            ];
        } else {
            $total_paid = 0;
            foreach ($request_data["transfers"] as $transfer) {
                if (in_array($transfer["hashid"], $current_payments)) {
                    continue;
                }

                $add_transaction_response = localAPI('AddTransaction', [
                    'paymentmethod' => $gateway_params['name'],
                    'invoiceid' => $invoice_id,
                    'userid' => $client_data["userid"],
                    'transid' => $transfer["hashid"],
                    'date' => (new DateTime($transfer["date"]))->format("d/m/Y"),
                    'amountin' => $transfer["amount"],
                    'allowduplicatetransid' => true,
                ]);

                if ($add_transaction_response["result"] !== "success") {
                    errorResponse("Ha ocurrido un error en la llamada de la API (AddTransaction)");
                }
                
                logTransaction($gateway_params['name'], $transfer, "Transaction added");

                $total_paid += $transfer["amount"];
            }

            $message = "El pago ha sido ingresado";
            $restart = false;

            // When using CLP there are no decimals, so we add the extra to complete the payment without rounding the balance
            if ($total_paid < doubleval($invoice_data["balance"])) {
                if ($request_data["completeness"] === "exact") {
                    addInvoicePayment($invoice_id, "Extra para completar el pago", doubleval($invoice_data["balance"]) - $total_paid, 0, $gateway_params['name']);
                    logTransaction($gateway_params['name'], [], "Extra transaction added");
                } elseif ($request_data["completeness"] === "underpaid") {
                    $message .= ", sin embargo aún tiene un saldo pendiente de $" . number_format($balance - $total_paid, 0, ',', '.');
                    $restart = true;
                }
            } elseif ($total_paid > $balance) {
                $message .= " y se han abonado $" . number_format($balance - $total_paid, 0, ',', '.') . " de crédito a su cuenta";
            }

            $response = [
                "status" => "accepted",
                "message" => $message,
                "redirect" => str_replace('%INVOICE_ID%', $invoice_id, $gateway_params['redirect']),
                "restart" => $restart,
            ];
        }
    } elseif ($action === "cancellation") {
        $current_payments = getCurrentPayments($invoice_data["transactions"]["transaction"]);
        $total_refund = 0;
        foreach ($request_data["transfers"] as $transfer) {
            if (!in_array($transfer["hashid"], $current_payments)) {
                continue;
            }

            $add_transaction_response = localAPI('AddTransaction', [
                'paymentmethod' => $gateway_params['name'],
                'invoiceid' => $invoice_id,
                'userid' => $client_data["userid"],
                'transid' => "refund-{$transfer["hashid"]}",
                'date' => (new DateTime($transfer["date"]))->format("d/m/Y"),
                'amountout' => $transfer["amount"],
                'allowduplicatetransid' => true,
            ]);

            if ($add_transaction_response["result"] !== "success") {
                errorResponse("Ha ocurrido un error en la llamada de la API (AddTransaction)");
            }

            logTransaction($gateway_params['name'], $transfer, "Refund transaction added");
            $total_refund += $transfer["amount"];
        }

        $message = "Reembolso agregado correctamente";
        $new_balance = $balance + $total_refund;

        // The refund may have made the invoice unpaid
        if ($new_balance > 0 && $invoice_data["status"] == "Paid") {
            $update_invoice_response = localAPI('UpdateInvoice', [
                'invoiceid' => $invoice_id,
                'status' => 'Unpaid',
            ]);

            if ($update_invoice_response["result"] !== "success") {
                errorResponse("Ha ocurrido un error en la llamada de la API (UpdateInvoice)");
            }

            $message .= ". Nuevo saldo: $" . number_format($new_balance, 0, ',', '.');
        }

        $response = ["message" => $message];
    } else {
        errorResponse("Acción inválida ($action)");
    }
} else {
    errorResponse("Tipo de request no soportado");
}

logModuleCall($gateway_module_name, $_SERVER["REQUEST_METHOD"], $request_data, $response, "", []);

echo json_encode($response);
