<?php
/**
* Plugin Name: Linkify
* Description: Linkify WooCommerce gateway to automatically validate payments by wire transfer between chilean banks.
* Version: 2.3.8
* Author: Linkify
* Author URI: https://www.linkify.cl
**/

include 'src/LinkifyEndpointsHandler.php';

defined('ABSPATH') or die('No script kiddies please!');

add_action('plugins_loaded', 'actionPluginsLoaded');
add_filter('woocommerce_payment_gateways', 'filterAddLinkifyGatewayClass');
add_filter('plugin_action_links_' . basename(__DIR__) . '/linkify_gateway.php', 'linkifySettingsLink');
add_action('init', 'registerPartiallyPaidOrderStatus');
add_filter('wc_order_statuses', 'addPartiallyPaidToOrderStatuses');
add_filter('woocommerce_valid_order_statuses_for_payment_complete', 'addPartiallyPaidToValidOrderStatusesForPaymentComplete');
         
// Hook to be called in the admin plugins page if Linkify is enabled but WooCommerce is not
function actionHandleNoBasePlugin()
{
    echo '<div class="error"> <p><strong>Es necesario tener instalado el plugin WooCommerce para usar Linkify.</strong></p></div>';
}

// Adds the settings links under the item on the plugins list (Ref: https://neliosoftware.com/blog/how-to-add-a-link-to-your-settings-in-the-wordpress-plugin-list/?nab=0)
function linkifySettingsLink($links)
{
    // Build and escape the URL
    $url = esc_url(add_query_arg(['page' => 'wc-settings', 'tab' => 'checkout', 'section' => 'linkify_gateway'], get_admin_url() . 'admin.php'));

    // Create the link
    $settings_link = '<a href="' . $url . '">' . __('Settings') . '</a>';

    // Adds the link to the start of the array
    array_unshift($links, $settings_link);

    return $links;
}

// Creates the partially paid order status
function registerPartiallyPaidOrderStatus()
{
    register_post_status('wc-partially-paid', [
        'label'                     => 'Parcialmente pagado',
        'public'                    => true,
        'show_in_admin_status_list' => true,
        'show_in_admin_all_list'    => true,
        'exclude_from_search'       => false,
        'label_count'               => _n_noop('Parcialmente pagado <span class="count">(%s)</span>', 'Parcialmente pagado <span class="count">(%s)</span>')
    ]);
}

// Adds the partially paid order status after the pending status to list of WC Order statuses
function addPartiallyPaidToOrderStatuses($order_statuses)
{
    $plugin_settings = get_option("woocommerce_linkify_gateway_settings", []);
    if ($plugin_settings["accept_underpaid"] != "yes") {
        return $order_statuses;
    }

    $new_order_statuses = [];

    foreach ($order_statuses as $key => $status) {
        $new_order_statuses[ $key ] = $status;
        if ('wc-pending' === $key) {
            $new_order_statuses['wc-partially-paid'] = 'Parcialmente pagado';
        }
    }
    
    return $new_order_statuses;
}

// Adds the partially paid order status to the valid statuses to mark and order as complete
function addPartiallyPaidToValidOrderStatusesForPaymentComplete($order_statuses)
{
    $order_statuses[] = 'partially-paid';

    return $order_statuses;
};

// Load plugin main class
function actionPluginsLoaded()
{
    // Check for Woocommerce installation. Display error if not
    if (!class_exists('WC_Payment_Gateway')) {
        add_action('admin_notices', 'actionHandleNoBasePlugin');
        return;
    }

    // Main gateway class
    class WC_Gateway_Linkify_Gateway extends WC_Payment_Gateway
    {
        public function __construct()
        {
            $this->dev_file = __DIR__ . "/dev.php";
            if (is_file($this->dev_file)) {
                include_once $this->dev_file;
            }

            $this->id = 'linkify_gateway';
            $this->app_url = $linkify_app_url ?? 'https://app.linkify.cl';
            $this->has_fields = false;
            $this->icon = 'https://www.linkify.cl/images/logos/woocommerce.png';
            $this->method_title = 'Linkify';
            $this->method_description = 'Valida transferencias electrónicas en Chile de manera automática.';

            $this->initFormFields();
            $this->init_settings(); // load get_options values from user settings

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->merchant_id = $this->get_option('merchant_id');
            $this->secret_key = $this->get_option('secret_key');
            $this->accept_overpaid = $this->get_option('accept_overpaid');
            $this->accept_underpaid = $this->get_option('accept_underpaid');
            $this->send_endpoint = $this->get_option('send_endpoint');
            $this->change_stock_on_partial_paymeny = $this->get_option('change_stock_on_partial_paymeny');
            
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
            add_action('woocommerce_receipt_' . $this->id, [$this, 'actionLinkifyReceipt']);
            add_action('woocommerce_api_linkify_gateway', [$this, 'actionEndpointHandler']);
            add_filter('woocommerce_settings_api_sanitized_fields_' .$this->id, [$this, 'actionHandleSanitizeSettings']);
        }
        
        /** START SECTION: PLUGIN SETTINGS **/
        // Override parent function in order to show error notices. Checks if merchant_id is set and if CLP is current currency
        public function admin_options()
        {
            ?>
            <script type="text/javascript">
                document.addEventListener("DOMContentLoaded", function(e) { // Wait for dom to be loaded
                    // Remove the success notice (has id message). It said that everything was saved successfully when really not. Dont know how else to get rid of it
                    let errors = document.getElementsByClassName("error");
                    if (errors.length > 0) { // If there are errors, don't display the success message.
                        let success_notice = document.getElementById("message");
                        while (success_notice != null) {
                            success_notice.parentNode.removeChild(success_notice);
                            success_notice = document.getElementById("message");
                        }
                    }  
                });
            </script>
            <?php $this->getOptionsErrors(); ?>
            <h2><?php _e('Linkify', 'woocommerce'); ?></h2>
            <table class="form-table">
            <?php $this->generate_settings_html(); ?>
            </table>
            <?php
        }
        
        // Check if the plugin is correctly setup, and display errors if not
        private function getOptionsErrors()
        {
            if ($this->merchant_id == '') { // When updating a wrong pair of merchant_id/secret_key this value should be an empty string, so we check for it to check for errors
                $this->displayError("No se han ingresado valores correctos de ID de cuenta y clave privada. Linkify no funcionará hasta que se ingresen.", "error");
            }

            if (!("CLP" === get_woocommerce_currency())) { // Show warning if CLP is not the current currency
                $this->displayError("Advertencia: Tu tienda no tiene seleccionado el peso chileno como moneda. Linkify no funcionará correctamente.", "update-nag");
            }
        }

        // error_type can be error (red error) or update-nag (yellow warning)
        // This funciont should show an error in the settings page of the plugin
        private function displayError($error_message, $error_type)
        {
            echo '<div id="woocommerce_errors" class="notice '. $error_type.'"><p><strong>'.$error_message.'</strong></p></div>';
        }

        // If credentials could not be verified with Linkify (wrong merchant_id or secret_key), this handle the user input in settings to override merchant_id, secret_key and enabled
        // This hook should prevent the user to input and save invalid data
        public function actionHandleSanitizeSettings($settings)
        {
            if ($this->checkLinkifyCredentialErrors($settings['merchant_id'], $settings['secret_key'])) {
                $settings['merchant_id'] = '';
                $settings['secret_key'] = '';
                $settings['enabled'] = 'no';
            }

            return $settings;
        }

        // Returns true if there is an error with provided credentials
        private function checkLinkifyCredentialErrors($merchant_id, $secret_key)
        {
            $endpoint = "$this->app_url/api/check_credentials/$merchant_id";
            $params = array('secret_key' => $secret_key);
            $url = $endpoint . '?' . http_build_query($params);

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);

            $response = json_decode(curl_exec($ch));
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

            if (is_file($this->dev_file) && $http_code == 403) {
                exit("DEV MODE: Enable access from this host and refresh.");
            }

            return $http_code != 200 || !$response->success;
        }
        
        // Especification of the settings that the user can change. Merchant_id and secret_key should be provided
        public function initFormFields()
        {
            $this->form_fields = [
                'enabled' => array(
                    'title' => __('Enable/Disable', 'woocommerce'),
                    'type' => 'checkbox',
                    'label' => __('Habilitar pagos con Linkify', 'woocommerce'),
                    'default' => 'no'
                ),
                'title' => [
                    'title' => __('Title', 'woocommerce'),
                    'type' => 'text',
                    'description' => __('Título que el usuario verá en la pantalla de checkout.', 'woocommerce'),
                    'default' =>  __('Linkify - Transferencia electrónica', 'woocommerce'),
                    'desc_tip' => true,
                ],
                'description' => [
                    'title' => __('Descripción', 'woocommerce'),
                    'type' => 'textarea',
                    'description' => __('Descripción del medio de pago que el usuario verá en el checkout'),
                    'default' => 'Pago automático por transferencia electrónica a bancos chilenos',
                    'desc_tip' => true,
                ],
                'merchant_id' => [
                    'title' => __('ID de cuenta', 'woocommerce'),
                    'type' => 'text',
                    'description' => __('ID de su cuenta en Linkify'),
                    'default' => '',
                    'desc_tip' => true,
                ],
                'secret_key' => [
                    'title' => __('Clave privada', 'woocommerce'),
                    'type' => 'password',
                    'description' => __('Clave privada de su cuenta en Linkify'),
                    'default' => '',
                    'desc_tip' => true,
                ],
                'accept_overpaid' => [
                    'title' => __('Aceptar pagos mayores al total', 'woocommerce'),
                    'type' => 'checkbox',
                    'label' => 'Recuerda habilitar esta opción en tu cuenta de Linkify también',
                    'description' => __('Permite pagar un cobro con una transferencia mayor al total. Por ejemplo, si hago un cobro por $5.000 y hacen una transferencia por $7.000, permite elegir esa transferencia para pagar el cobro. Linkify pide confirmación al pagador de que quiere usar esa transferencia que es mayoral total. Es responsabilidad del cobrador manejar las devoluciones en estos casos.', 'woocommerce'),
                    'default' => 'no',
                    'desc_tip' => true,
                ],
                'accept_underpaid' => [
                    'title' => __('Aceptar pagos menores al total', 'woocommerce'),
                    'type' => 'checkbox',
                    'label' => 'Recuerda habilitar esta opción en tu cuenta de Linkify también',
                    'description' => __('Permite pagar parcialmente un cobro. Por ejemplo, si hago un cobro por $5.000 y hacen una transferencia por $1.000, permite elegir esa transferencia para pagar parcialmente el cobro. Linkify pide confirmación al pagador de que quiere usar esa transferencia que es menor al total.', 'woocommerce'),
                    'default' => 'no',
                    'desc_tip' => true,
                ],
                'auto_redirect' => [
                    'title' => __('Redirigir automáticamente a Linkify', 'woocommerce'),
                    'type' => 'checkbox',
                    'label' => '',
                    'description' => __('Redirige automáticamente a Linkify luego de aceptar la orden', 'woocommerce'),
                    'default' => 'no',
                    'desc_tip' => true,
                ],
                'send_endpoint' => [
                    'title' => __('Enviar el endpoint en la URL del pago', 'woocommerce'),
                    'type' => 'checkbox',
                    'label' => ' ',
                    'description' => __('Utilizar en caso de que la URL de integración en Linkify no corresponda a la URL de esta instalación de WooCommerce', 'woocommerce'),
                    'default' => 'no',
                    'desc_tip' => true,
                ],
                'change_stock_on_partial_paymeny' => [
                    'title' => __('Modificar stock con pagos parciales', 'woocommerce'),
                    'type' => 'checkbox',
                    'label' => ' ',
                    'description' => __('Se modificará el stock al recibir o anular pagos parciales', 'woocommerce'),
                    'default' => 'no',
                    'desc_tip' => true,
                ]];
        }
        /** END SECTION: PLUGIN SETTINGS **/
            

        /** START SECTION: API
         * Functions to handle communication with Linkify.**/

        // Handle API calls from Linkify. Handles GET and POST requests
        public function actionEndpointHandler()
        {
            $end_point_handler = new LinkifyEndpointsHandler($this->secret_key, $this->accept_overpaid, $this->accept_underpaid, $this->change_stock_on_partial_paymeny);
            $end_point_handler->handleRequest($this);
        }

        /** END SECTION: API **/
        
        // Show Linkify link to pay in receipt page of an order
        public function actionLinkifyReceipt($order_id)
        {
            $error = 'No se puede pagar. Linkify no se encuentra habilitado. Contáctese con el administrador del sitio.';
            if (!$this->checkClpCurrencyEnabled($order_id)) {
                $error = 'Linkify no está habilitado para esta moneda. Elija otro medio de pago.';
            } elseif ($this->is_available()) {
                $merchant_id = $this->get_option('merchant_id');
                $linkify_url = "$this->app_url/pay/$merchant_id/new/remote/$order_id";

                if ($this->send_endpoint == "yes") {
                    $endpoint = base64_encode(get_site_url() . "/wc-api/linkify_gateway");
                    $linkify_url .= "?endpoint=$endpoint";
                }

                if($this->get_option('auto_redirect') == 'yes') {
                  header("Location: $linkify_url");
                  die();
                } else {
                  echo '<a href="' . $linkify_url . '"><button> Ir a pagar </button> </a>';
                }

                return;
            }

            echo "<div><p><strong>$error</strong></p></div>";
        }
        
        // Check if clp is the current currency. If not, return false. Currency is got in different ways in woocommerce 2 vs 3. So we try both ways
        private function checkClpCurrencyEnabled($order_id)
        {
            $order = new WC_Order($order_id);
            if (method_exists($order, 'get_currency')) {
                return ($order->get_currency() == 'CLP');
            }

            return ($order->get_order_currency() == 'CLP');
        }
        
        // Linkify does not accept refunds. Should be handled by the store owner
        // This is an override of a parent method
        public function can_refund_order($order)
        {
            return false;
        }
        
        // Tells whether the plugin needs setup or not. Check for correct setting of merchant_id and secret key
        // If this returns false woocommerce should disable linkify and not let the user enable it until needs_setup returns true
        // This is an override of a parent method
        public function needs_setup()
        {
            return $this->get_option('merchant_id') == '' || $this->get_option('secret_key') == '';
        }
        
        // Process payment. Redirect to Linkify receipt page in order to give him instructions on how to pay
        // This is an override of a parent method
        public function process_payment($order_id)
        {
            global $woocommerce;
            $order = new WC_Order($order_id);
            $woocommerce->cart->empty_cart();
            return ['result' => 'success', 'redirect' =>$order->get_checkout_payment_url(true)];
        }
    }
}

// Add linkify class as a woocommerce gateway class
function filterAddLinkifyGatewayClass($methods)
{
    $methods[] = 'WC_Gateway_Linkify_Gateway';
    return $methods;
}
